<?php

namespace App\Http\Controllers;

use App\Models\HiddenBacklink; // Changed model reference from FooterBacklink to HiddenBacklink
use App\Models\HiddenAnchor;
use Illuminate\Http\Request;

class HiddenBacklinkController extends Controller
{
    public function index()
    {
        $backlinks = HiddenBacklink::with('anchors')->get(); // Updated to use HiddenBacklink
        return view('hidden_backlinks.index', compact('backlinks'));
    }

    public function create()
    {
        return view('hidden_backlinks.create');
    }

    public function store(Request $request)
    {
        try {
            $request->validate([
                'title' => 'required|string|max:255',
                'website_url' => 'required|url|max:255',
                'anchors.*.anchor_text' => 'required|string|max:255',
                'anchors.*.domain' => 'required|string|max:2550',
            ]);
    
            $backlink = HiddenBacklink::create([ 
                'title' => $request->title,
                'website_url' => $request->website_url,
            ]);
    
            foreach ($request->anchors as $anchor) {
                $backlink->anchors()->create($anchor);
            }
    
            return redirect()->route('hidden_backlinks.index')->with('success', 'Hidden backlink added successfully.');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    public function edit(HiddenBacklink $hiddenBacklink) // Changed to HiddenBacklink
    {
        return view('hidden_backlinks.edit', compact('hiddenBacklink'));
    }

    public function update(Request $request, HiddenBacklink $hiddenBacklink) // Changed to HiddenBacklink
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'anchors.*.anchor_text' => 'required|string|max:255',
            'anchors.*.domain' => 'required|string|max:2550',
        ]);

        $hiddenBacklink->update(['title' => $request->title]);

        $hiddenBacklink->anchors()->delete();
        foreach ($request->anchors as $anchor) {
            $hiddenBacklink->anchors()->create($anchor);
        }

        return redirect()->route('hidden_backlinks.index')->with('success', 'Hidden backlink updated successfully.');
    }

    public function destroy(HiddenBacklink $hiddenBacklink) // Changed to HiddenBacklink
    {
        $hiddenBacklink->delete();
        return redirect()->route('hidden_backlinks.index')->with('success', 'Hidden backlink deleted successfully.');
    }

    public function getAnchorsByDomain(Request $request)
    {
        // Validate the domain input (query parameter)
        $request->validate([
            'domain' => 'required|string|max:2550',
        ]);

        $domain = parse_url($request->query('domain'), PHP_URL_HOST) ?: $request->query('domain');

        // Get all backlinks where anchors' domain matches the provided domain
        $backlinks = HiddenBacklink::whereHas('anchors', function ($query) use ($domain) { // Changed to HiddenBacklink
            $query->where('domain', 'like', '%' . $domain . '%');
        })
        ->with(['anchors' => function($query) use ($domain) {
            // Filter anchors by matching the domain
            $query->where('domain', 'like', '%' . $domain . '%');
        }])
        ->get();

        // Extract website URLs and the first anchor text that matches the domain
        $data = $backlinks->map(function ($backlink) {
            // Get the first anchor text (if any)
            $firstAnchorText = $backlink->anchors->first()->anchor_text ?? null;

            return [
                'website_url' => $backlink->website_url,
                'anchors' => $firstAnchorText, // Just the first anchor text as a string
            ];
        });

        // Return only the anchors_data directly as a JSON array
        return response()->json($data);
    }
}
