<?php

namespace App\Http\Controllers;

use App\Models\Slide;
use App\Models\ListItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Spatie\Activitylog\Models\Activity;

class SlideController extends Controller
{
    public function create()
{
    return view('slides.create');
}
    // Display a listing of the slides
    public function index()
{
    $slides = Slide::with('listItems', 'creator', 'updater')
        ->orderBy('created_at', 'desc') // Order by creation date in descending order
        ->paginate(10);

    return view('slides.index', compact('slides'));
}

    // Store a newly created slide in storage


    // Display the specified slide
    public function show(Slide $slide)
    {
 
    $allSlides = Slide::all(); // Fetch all slides for the dropdown

    return view('slides.show', compact('slide', 'allSlides'));
    }

    // Show the form for editing the specified slide
    public function edit(Slide $slide)
    {
        return view('slides.edit', compact('slide'));
    }

    // Update the specified slide in storage
public function store(Request $request)
{
    $validated = $request->validate([
        'group_name' => 'required|string',
        'heading' => 'required|string',
        'list_items' => 'required|array',
        'list_items.*.list_name' => 'required|string',
        'list_items.*.domains' => 'required|string',
    ]);

    $slide = Slide::create([
        'group_name' => $validated['group_name'],
        'heading' => $validated['heading'],
        'created_by' => Auth::id(), // Ensure this is set
    ]);

    foreach ($validated['list_items'] as $listItem) {
        $slide->listItems()->create([
            'list_name' => $listItem['list_name'],
            'domains' => $listItem['domains'],
        ]);
    }

    return redirect()->route('slides.index')->with('success', 'Slide created successfully.');
}

public function update(Request $request, Slide $slide)
{
    $validated = $request->validate([
        'group_name' => 'required|string',
        'heading' => 'required|string',
        'list_items' => 'required|array',
        'list_items.*.list_name' => 'required|string',
        'list_items.*.domains' => 'required|string',
    ]);

    $slide->update(array_merge($validated, [
        'updated_by' => Auth::id(),
    ]));

    $slide->listItems()->delete();
    foreach ($validated['list_items'] as $listItem) {
        $slide->listItems()->create($listItem);
    }

    activity()
        ->causedBy(Auth::user())
        ->performedOn($slide)
        ->log("Updated slide: {$slide->group_name}");

    return redirect()->route('slides.index')->with('success', 'Slide updated successfully.');
}


    // Remove the specified slide from storage
    public function destroy(Slide $slide)
    {
        $deletedSlide = $slide->toArray(); // Save slide data for logging
        $slide->delete();

        // Log the deletion of the slide
        activity()
            ->causedBy(Auth::user())
            ->performedOn($slide)
            ->log("Deleted slide: {$deletedSlide['group_name']}");

        return redirect()->route('slides.index')->with('success', 'Slide moved to trash.');
    }

    // Permanently delete the specified slide
    public function forceDelete(Slide $slide)
{
   
    $slide->forceDelete();

    activity()
        ->causedBy(Auth::user())
        ->performedOn($slide)
        ->log("Permanently deleted slide: {$slide->group_name}");

    return redirect()->route('slides.trash')->with('success', 'Slide permanently deleted.');
}


    // Restore the specified slide
    public function restore($id)
    {
        $slide = Slide::withTrashed()->findOrFail($id);
        $slide->restore();

        // Log the restoration of the slide
        activity()
            ->causedBy(Auth::user())
            ->performedOn($slide)
            ->log("Restored slide: {$slide->group_name}");

        return redirect()->route('slides.trash')->with('success', 'Slide restored.');
    }

    // Display the trash with soft-deleted slides
    public function trash()
{
    
    $slides = Slide::onlyTrashed()->paginate(10);
    return view('slides.trash', compact('slides'));
}


    // Display the history of slide activities
    public function history()
    {
        $activities = Activity::where('subject_type', 'App\Models\Slide')->latest()->get();
        return view('slides.history', compact('activities'));
    }

public function fetchSlides()
{
    // Fetch slides with their list items
    $slides = Slide::with('listItems')->get();

    // Return the data as a JSON response
    return response()->json($slides);
}
// Move a list item to another slide
   public function move(Request $request)
{
    // Find the list item by its ID
    $listItem = ListItem::findOrFail($request->list_item_id);

    // Update the slide_id to the new slide
    $listItem->slide_id = $request->slide_id;
    $listItem->save();

    // Redirect to the show page of the moved-to slide with a success message
    return redirect()->route('slides.show', $request->slide_id)
                     ->with('success', 'List item moved successfully.');
}


    
}

